<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Http\Response;
use Cake\I18n\FrozenTime;
use Cake\Log\Log;
use Cake\ORM\Query;
use Cake\ORM\Table;
use Cake\ORM\TableRegistry;

class CalendarController extends AppController
{
    /** @var Table */
    private Table $CalendarEvents;
    /** @var Table */
    private Table $Login;

    // ============================
    // DO NOT DELETE: helpers used to scope everything to the current clinic
    // ============================

    public function initialize(): void
    {
        parent::initialize();
        $this->loadComponent('Authentication.Authentication');
        $this->CalendarEvents = $this->fetchTable('CalendarEvents');
        $this->Login          = $this->fetchTable('Login');
        $this->viewBuilder()->setLayout('default');
    }

    public function beforeFilter(\Cake\Event\EventInterface $event)
    {
        parent::beforeFilter($event);

        if ($this->components()->has('Authentication')) {
            $result = $this->Authentication->getResult();
            if (!$result || !$result->isValid()) {
                return $this->redirect([
                    'controller' => 'Login',
                    'action'     => 'index',
                    '?'          => ['redirect' => $this->request->getRequestTarget()],
                ]);
            }
        }
    }

    public function index()
    {
        // ============================
        // DO NOT DELETE: filter clinician list to current clinic
        // ============================
        $clinicId = $this->currentClinicIdOrNull();

        $logins = $this->Login
            ->find()
            ->select(['Login.id','Login.first_name','Login.last_name','Login.email'])
            ->innerJoin(
                ['cu' => 'clinic_users'],
                'cu.login_id = Login.id'
            );

        if ($clinicId) {
            $logins->where(['cu.clinic_id' => $clinicId]);
        }

        $list = [];
        foreach ($logins->orderAsc('Login.first_name') as $r) {
            $name = trim(($r->first_name ?? '') . ' ' . ($r->last_name ?? ''));
            $list[$r->id] = $name !== '' ? $name : ($r->email ?? '');
        }

        $this->set('clinicians', $list);
    }

    public function feed(): Response
    {
        $this->request->allowMethod(['get']);

        $clinicId = $this->currentClinicIdOrNull();
        $startQ   = $this->request->getQuery('start');
        $endQ     = $this->request->getQuery('end');
        $cid      = $this->request->getQuery('clinician_id');

        $q = $this->CalendarEvents->find()->orderAsc('start');

        if ($startQ) $q->where(['start >=' => new FrozenTime($startQ)]);
        if ($endQ)   $q->where(['start <=' => new FrozenTime($endQ)]);
        if ($cid !== null && $cid !== '') $q->where(['user_id' => (int)$cid]);

        // ============================
        // DO NOT DELETE: scope by clinic
        // ============================
        $this->applyClinicScope($q, $clinicId);

        $statusColors = [
            'confirmed' => '#63ed7a',
            'pending'   => '#ffa426',
            'cancelled' => '#fc544b',
        ];

        $out = [];
        foreach ($q as $e) {
            $bg = $e->bg_color ?: ($statusColors[$e->status] ?? null);
            $ext = [
                'status'           => $e->status,
                'location'         => $e->location,
                'user_id'          => $e->user_id,
                'appointment_type' => $e->appointment_type,
                'service_type'     => $e->service_type ?? '',
                'service_id'   => $e->service_id   ?? null,
                'service_name' => $e->service_name ?? ($e->service_type ?? ''),
            ];
            if ($this->CalendarEvents->hasField('contact'))        $ext['contact'] = $e->contact ?? '';
            if ($this->CalendarEvents->hasField('participant_id')) $ext['participant_id'] = $e->participant_id ?? null;

            $out[] = [
                'id'    => (string)$e->id,
                'title' => (string)($e->title ?? ''),
                'start' => $e->start?->i18nFormat("yyyy-MM-dd'T'HH:mm:ss"),
                'end'   => $e->end?->i18nFormat("yyyy-MM-dd'T'HH:mm:ss"),
                'backgroundColor' => $bg,
                'borderColor'     => $bg,
                'textColor'       => $e->text_color ?: null,
                'extendedProps'   => $ext,
            ];
        }

        return $this->response->withType('json')->withStringBody(json_encode($out));
    }

    private function readJson(): array
    {
        $raw = (string)$this->request->getBody();
        $data = json_decode($raw, true);
        return is_array($data) ? $data : (array)$this->request->getData();
    }

    public function store(): Response
    {
        $this->request->allowMethod(['post']);

        try {
            $data = $this->readJson();

            $e = $this->CalendarEvents->newEmptyEntity();

            $e->title   = (string)($data['title'] ?? 'Untitled');
            $e->start   = new FrozenTime((string)($data['start'] ?? 'now'));
            $e->end     = !empty($data['end']) ? new FrozenTime((string)$data['end']) : null;
            $e->status  = (string)($data['status'] ?? 'pending');
            $e->user_id = isset($data['user_id']) && $data['user_id'] !== '' ? (int)$data['user_id'] : null;
            $e->location    = (string)($data['location'] ?? '');
            $e->description = (string)($data['description'] ?? '');
            $e->appointment_type = in_array(($data['appointment_type'] ?? 'standard'), ['first','standard'], true)
                ? (string)$data['appointment_type'] : 'standard';
            $e->service_type = (string)($data['service_type'] ?? '');
            $e->service_id   = isset($data['service_id']) && $data['service_id'] !== '' ? (int)$data['service_id'] : null;
            if ($this->CalendarEvents->hasField('service_name')) {
                $e->service_name = (string)($data['service_name'] ?? $data['service'] ?? $data['service_type'] ?? '');
            }
            if ($this->CalendarEvents->hasField('contact')) {
                $e->contact = (string)($data['contact'] ?? '');
            }

            if ($this->CalendarEvents->hasField('participant_id')) {
                $pid = $data['participant_id'] ?? null;
                $e->participant_id = ($pid === '' || $pid === null) ? null : (int)$pid;
            }

            if (!$this->CalendarEvents->save($e)) {
                return $this->response->withStatus(422)->withType('json')
                    ->withStringBody(json_encode([
                        'error'  => 'Create failed',
                        'errors' => $e->getErrors(),
                    ]));
            }

            return $this->response->withType('json')
                ->withStringBody(json_encode(['ok' => true, 'id' => (string)$e->id]));

        } catch (\Throwable $ex) {
            Log::error('Calendar store failed: ' . $ex->getMessage());
            return $this->response->withStatus(500)->withType('json')
                ->withStringBody(json_encode(['error' => $ex->getMessage()]));
        }
    }

    public function move(int $id): Response
    {
        $this->request->allowMethod(['post']);

        $e = $this->CalendarEvents->get($id);
        $payload = (array)$this->request->getData();

        $start = $payload['start'] ?? null;
        $end   = array_key_exists('end', $payload) ? $payload['end'] : null;

        if ($start !== null && $start !== '') $e->start = new FrozenTime((string)$start);
        $e->end = ($end === null || $end === '') ? null : new FrozenTime((string)$end);

        $this->CalendarEvents->save($e);

        return $this->response->withType('json')->withStringBody(json_encode(['ok' => true]));
    }

    public function status(int $id): Response
    {
        $this->request->allowMethod(['post']);
        $e  = $this->CalendarEvents->get($id);
        $st = (string)($this->request->getData('status') ?? '');

        if (!in_array($st, ['confirmed','pending','cancelled'], true)) {
            return $this->response->withStatus(422)
                ->withStringBody(json_encode(['error' => 'Invalid status']));
        }
        $e->status = $st;
        $this->CalendarEvents->save($e);

        return $this->response->withType('json')->withStringBody(json_encode(['ok' => true]));
    }

    public function clinicians(): \Cake\Http\Response
    {
        $this->request->allowMethod(['get']);

        $clinicId = $this->currentClinicIdOrNull();

        $q = $this->Login->find()
            ->select(['id','first_name','last_name','email'])
            ->orderAsc('first_name');

        if ($clinicId) {
            $q->innerJoin(['cu' => 'clinic_users'], 'cu.login_id = Login.id')
                ->where(['cu.clinic_id' => $clinicId]);
        }

        $rows = $q->distinct(['Login.id'])->all();
        $out = [];
        foreach ($rows as $r) {
            $name = trim(($r->first_name ?? '') . ' ' . ($r->last_name ?? ''));
            $out[] = [
                'id'   => (int)$r->id,
                'name' => $name !== '' ? $name : (string)($r->email ?? ''),
            ];
        }
        return $this->response->withType('json')->withStringBody(json_encode($out));
    }

    public function events(): Response
    {
        $this->request->allowMethod(['get']);

        $clinicId = $this->currentClinicIdOrNull();  // DO NOT DELETE
        $startQ   = $this->request->getQuery('start');
        $endQ     = $this->request->getQuery('end');
        $cid      = $this->request->getQuery('clinician_id');

        $start = $startQ ? new FrozenTime((string)$startQ) : null;
        $end   = $endQ   ? new FrozenTime((string)$endQ)   : null;

        $q = $this->CalendarEvents->find()
            ->contain(['Login'])
            ->orderAsc('start');

        if ($start) $q->where(['start >=' => $start]);
        if ($end)   $q->where(['start <=' => $end]);
        if ($cid !== null && $cid !== '') $q->where(['user_id' => (int)$cid]);

        // ============================
        // DO NOT DELETE: scope by clinic
        // ============================
        $this->applyClinicScope($q, $clinicId);

        $statusColors = ['confirmed'=>'#63ed7a','pending'=>'#ffa426','cancelled'=>'#fc544b'];

        $out = [];
        foreach ($q as $e) {
            $bg = $e->bg_color ?: ($statusColors[$e->status] ?? null);

            $xp = [
                'status'          => $e->status,
                'clinician'       => trim(($e->login->first_name ?? '') . ' ' . ($e->login->last_name ?? '')),
                'user_id'         => $e->user_id,
                'location'        => (string)$e->location,
                'description'     => (string)$e->description,
                'appointment_type'=> $e->appointment_type,
                'service_type'    => $e->service_type ?? '',
                'service_id'      => $e->service_id ?? null,
            ];
            if ($this->CalendarEvents->hasField('contact'))        $xp['contact'] = (string)($e->contact ?? '');
            if ($this->CalendarEvents->hasField('participant_id')) $xp['participant_id'] = $e->participant_id ?? null;

            $out[] = [
                'id'    => (string)$e->id,
                'title' => (string)$e->title,
                'start' => $e->start?->i18nFormat("yyyy-MM-dd'T'HH:mm:ss"),
                'end'   => $e->end?->i18nFormat("yyyy-MM-dd'T'HH:mm:ss"),
                'backgroundColor' => $bg,
                'borderColor'     => $bg,
                'extendedProps'   => $xp,
            ];
        }
        return $this->response->withType('json')->withStringBody(json_encode($out));
    }

    public function update(int $id): Response
    {
        $this->request->allowMethod(['post','patch','put']);
        $e = $this->CalendarEvents->get($id);
        $d = $this->readJson();

        if (array_key_exists('title', $d))       $e->title       = (string)$d['title'];
        if (array_key_exists('status', $d))      $e->status      = (string)$d['status'];
        if (array_key_exists('user_id', $d))     $e->user_id     = ($d['user_id'] === '' ? null : (int)$d['user_id']);
        if (array_key_exists('location', $d))    $e->location    = (string)$d['location'];
        if (array_key_exists('description', $d)) $e->description = (string)$d['description'];

        if (array_key_exists('start', $d)) {
            $e->start = $d['start'] ? new FrozenTime((string)$d['start']) : $e->start;
        }
        if (array_key_exists('end', $d)) {
            $e->end = ($d['end'] === '' || $d['end'] === null) ? null : new FrozenTime((string)$d['end']);
        }
        if (array_key_exists('appointment_type', $d)) {
            $e->appointment_type = in_array($d['appointment_type'], ['first','standard'], true)
                ? (string)$d['appointment_type'] : 'standard';
        }

        if (array_key_exists('service_type', $d)) $e->service_type = (string)$d['service_type'];
        if (array_key_exists('service_id', $d)) {
            $e->service_id = ($d['service_id'] === '' || $d['service_id'] === null) ? null : (int)$d['service_id'];
        }

        if ($this->CalendarEvents->hasField('service_name') && array_key_exists('service_name', $d)) {
            $e->service_name = (string)($d['service_name'] ?? '');
        }

        if ($this->CalendarEvents->hasField('contact') && array_key_exists('contact', $d)) {
            $e->contact = (string)($d['contact'] ?? '');
        }
        if ($this->CalendarEvents->hasField('participant_id') && array_key_exists('participant_id', $d)) {
            $pid = $d['participant_id'];
            $e->participant_id = ($pid === '' || $pid === null) ? null : (int)$pid;
        }

        if (!$this->CalendarEvents->save($e)) {
            return $this->response->withStatus(422)->withType('json')
                ->withStringBody(json_encode(['error'=>'Update failed','errors'=>$e->getErrors()]));
        }
        return $this->response->withType('json')->withStringBody(json_encode(['ok'=>true]));
    }

    public function delete(int $id): Response
    {
        $this->request->allowMethod(['post']);
        $e = $this->CalendarEvents->get($id);

        if ($this->CalendarEvents->delete($e)) {
            return $this->response->withType('json')
                ->withStringBody(json_encode(['ok' => true]));
        }

        return $this->response->withStatus(422)->withType('json')
            ->withStringBody(json_encode(['error' => 'Delete failed']));
    }

    public function locations(): Response
    {
        $this->request->allowMethod(['get']);

        $clinicId = $this->currentClinicIdOrNull();
        if (!$clinicId) {
            return $this->response->withStatus(400)->withType('json')
                ->withStringBody(json_encode(['error' => 'No clinic selected']));
        }

        $ClinicLocations = $this->fetchTable('ClinicLocations');

        $rows = $ClinicLocations->find()
            ->where(['clinic_id' => $clinicId])
            ->order(['is_primary' => 'DESC', 'id' => 'ASC'])
            ->all();

        $out = [];
        foreach ($rows as $r) {
            $label = trim(implode(', ', array_filter([
                $r->street,
                $r->suburb,
                $r->state,
                $r->postcode,
                $r->country,
            ])));
            if ($label === '') continue;

            $out[] = [
                'id'     => (string)$r->id,
                'label'  => $label,
                'primary'=> (int)$r->is_primary === 1,
            ];
        }

        return $this->response->withType('json')
            ->withStringBody(json_encode($out));
    }

    private function currentClinicIdOrNull(): ?int
    {
        if (method_exists($this, 'currentClinicId')) {
            return (int)($this->currentClinicId() ?? 0) ?: null;
        }
        $cid = $this->request->getSession()->read('Current.clinic_id');
        return $cid ? (int)$cid : null;
    }

    private function tableExists(string $name): bool
    {
        try {
            $locator = TableRegistry::getTableLocator();
            $locator->get($name); // throws if not resolvable
            $tables = $locator->get($name)->getConnection()->getSchemaCollection()->listTables();
            return in_array($locator->get($name)->getTable(), $tables, true);
        } catch (\Throwable $e) {
            return false;
        }
    }

    private function applyClinicScope(Query $q, ?int $clinicId): Query
    {
        if (!$clinicId) {
            return $q;
        }

        $events = $this->CalendarEvents;
        $hasParticipantId = $events->hasField('participant_id');
        $hasPCLink        = $this->tableExists('ParticipantsClinics');

        $q->leftJoin(
            ['cu' => 'clinic_users'],
            'cu.login_id = ' . $events->aliasField('user_id')
        );

        $or = [
            'cu.clinic_id' => $clinicId,
        ];

        if ($hasParticipantId && $hasPCLink) {
            $q->leftJoin(
                ['pc' => 'participants_clinics'],
                'pc.participant_id = ' . $events->aliasField('participant_id')
            );
            $or[] = ['pc.clinic_id' => $clinicId];
        }

        $q->where(['OR' => $or]);

        return $q;
    }
}
