<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Event\EventInterface;
use Cake\Http\Response;
use Cake\I18n\FrozenDate;
use Cake\I18n\FrozenTime;
use Cake\Mailer\Mailer;
use Cake\Routing\Router;
use Cake\Utility\Security;

class ClinicianRegisterController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        $this->loadComponent('Flash');
        $this->loadComponent('Authentication.Authentication');

        $this->viewBuilder()->setLayout('default');

        // Tables we’ll use
        $this->Login       = $this->fetchTable('Login');
        $this->ClinicUsers = $this->fetchTable('ClinicUsers');
        $this->Specializations = $this->fetchTable('Specializations');
    }

    public function beforeFilter(EventInterface $event)
    {
        parent::beforeFilter($event);

        $result   = $this->Authentication->getResult();
        $identity = $this->request->getAttribute('identity');
        $role     = strtolower((string)($identity->get('role') ?? ''));

        if (!$result || !$result->isValid()) {
            $this->Flash->error('Please login first.');
            $event->stopPropagation();
            $this->redirect(['controller' => 'Login', 'action' => 'index']);
            return;
        }

        if ($role !== 'admin') {
            $this->Flash->error('Only administrators can create accounts.');
            $event->stopPropagation();
            $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
            return;
        }
    }

    private function normalizeDob(?string $input): ?FrozenDate
    {
        $s = trim((string)$input);
        if ($s === '') return null;
        if (strpos($s, '/') !== false) {
            [$d,$m,$y] = array_pad(explode('/', $s, 3), 3, null);
            if ($d && $m && $y) {
                return new FrozenDate(sprintf('%04d-%02d-%02d', (int)$y, (int)$m, (int)$d));
            }
        }
        return new FrozenDate($s);
    }

    /**
     * Admin creates a new clinician in the **current clinic**.
     */
    public function index()
    {
        if ($resp = $this->requireAdmin()) { return $resp; }

        // We must have a current clinic context
        $clinicId = $this->currentClinicId();
        if (!$clinicId) {
            $this->Flash->error('No current clinic selected. Please re-login or contact support.');
            return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
        }

        // For the form
        $specializations = $this->Specializations->find('list', [
            'keyField'   => 'id',
            'valueField' => 'name',
        ])->orderAsc('name')->toArray();

        $preSelectedIds = (array)($this->request->getData('specializations._ids') ?? []);
        $this->set(compact('specializations', 'preSelectedIds'));

        // Show form
        if ($this->request->is('get')) {
            return;
        }

        $this->request->allowMethod(['post']);

        $raw = (array)$this->request->getData();

        // Basic password checks
        if (($raw['password'] ?? '') !== ($raw['password_confirm'] ?? '')) {
            $this->Flash->error('Passwords do not match.');
            return;
        }
        $pw = (string)($raw['password'] ?? '');
        if (!(preg_match('/[A-Z]/', $pw) && preg_match('/\d/', $pw) && strlen($pw) >= 8)) {
            $this->Flash->error('Password must be at least 8 characters and contain at least 1 uppercase letter and 1 number.');
            return;
        }

        $specIds = array_map('intval', (array)($raw['specializations']['_ids'] ?? []));
        $data = [
            'first_name'     => trim((string)($raw['first_name'] ?? '')),
            'last_name'      => trim((string)($raw['last_name'] ?? '')),
            'email'          => trim((string)($raw['email'] ?? '')),
            'mobile_phone'   => trim((string)($raw['mobile_phone'] ?? '')),
            'date_of_birth'  => $this->normalizeDob($raw['date_of_birth'] ?? '') ?: null,
            'password'       => $pw,
            'terms_agreed'   => 1,
            'role'           => 'clinician',
            'specializations'=> ['_ids' => $specIds],
        ];

        $entity = $this->Login->patchEntity($this->Login->newEmptyEntity(), $data, [
            'associated' => ['Specializations'],
        ]);

        if ($errors = $entity->getErrors()) {
            $this->Flash->error('The input information is incorrect. Please try again.');
            $this->set('errors', $errors);
            return;
        }

        // Save clinician + link to current clinic in one transaction
        $conn = $this->Login->getConnection();
        $conn->begin();

        try {
            $this->Login->saveOrFail($entity);

            // Create (or ensure) the clinic link
            $exists = $this->ClinicUsers->find()
                ->select(['id'])
                ->where(['clinic_id' => $clinicId, 'login_id' => (int)$entity->id])
                ->first();

            if (!$exists) {
                $link = $this->ClinicUsers->patchEntity(
                    $this->ClinicUsers->newEmptyEntity(),
                    [
                        'clinic_id' => $clinicId,
                        'login_id'  => (int)$entity->id,
                        'role'      => 'clinician',
                    ]
                );
                $this->ClinicUsers->saveOrFail($link);
            }

            $conn->commit();

            // Try to send the reset email (non-fatal if it fails)
            try {
                $this->sendResetEmailForNewClinician($entity);
                $this->Flash->success('Clinician account has been created. An email to set the password has been sent.');
            } catch (\Throwable $mailErr) {
                $this->Flash->success('Clinician account has been created, but the email could not be sent.');
            }

            return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
        } catch (\Throwable $e) {
            $conn->rollback();
            $this->Flash->error('Creation failed. Please try again.');
            return;
        }
    }

    private function requireAdmin(): ?Response
    {
        $result   = $this->Authentication->getResult();
        $identity = $this->request->getAttribute('identity');

        if (!$result || !$result->isValid() || !$identity) {
            $this->Flash->error('Please login first.');
            return $this->redirect(['controller' => 'Login', 'action' => 'index']);
        }

        $role = strtolower((string)($identity->get('role') ?? ''));
        if ($role !== 'admin') {
            $this->Flash->error('Only administrators can create accounts.');
            return $this->redirect(['controller' => 'Calendar', 'action' => 'index']);
        }

        return null;
    }

    private function sendResetEmailForNewClinician(\Cake\Datasource\EntityInterface $user): void
    {
        $token = bin2hex(random_bytes(32));
        $hmac  = $this->hmac($token);
        $user->set('reset_token', $hmac);
        $user->set('reset_token_expires', FrozenTime::now()->addHours(48));
        // persist just the reset bits
        $this->Login->saveOrFail($user, ['checkRules' => false, 'validate' => false]);

        $resetUrl = Router::url(['controller' => 'Login', 'action' => 'reset', $token], true);

        $toEmail = (string)$user->get('email');
        $toName  = trim(((string)$user->get('first_name')) . ' ' . ((string)$user->get('last_name'))) ?: null;

        $subject = 'Set your Co-Linic AI password';
        $expires = $user->get('reset_token_expires') instanceof FrozenTime
            ? $user->get('reset_token_expires')->i18nFormat('yyyy-MM-dd HH:mm')
            : 'soon';

        $text = "Hi " . ($toName ?: $toEmail) . ",\n\n"
            . "An administrator has created a Co-Linic AI account for you.\n"
            . "Set your password using this link:\n{$resetUrl}\n\n"
            . "This link may expire {$expires}. If it expires, use 'Forgot password' on the sign-in page to request a new link.\n\n"
            . "— Co-Linic AI\n";

        (new Mailer('default'))
            ->setTo($toEmail, $toName)
            ->setSubject($subject)
            ->setEmailFormat('text')
            ->deliver($text);
    }

    private function hmac(string $token): string
    {
        return hash_hmac('sha256', $token, (string)Security::getSalt());
    }
}
